/**
 * Copyright (c) 2017 Bosch SoftTec GmbH All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Bosch
 * SoftTec GmbH ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Bosch SoftTec (BSOT).
 *
 * BSOT MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. BSOT SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

#import <Foundation/Foundation.h>
#import "WDWPubDefinitions.h"

@class WDWClient, WDWWarning;


/**
 * Delegate protocol to interact with the WDW-SDK.
 * Delegate methods are only called when WDW-SDK monitoring is started.
 *
 * After the -[WDWClient startMonitoring] is called, the following delegate methods are called at arbitrary time intervals:
 *
 * + `-wdwClient:didReceiveError:`
 * + `-wdwClient:didReceiveDrivingWarning:`
 *
 * Other delegate methods are executed in the following order:
 *
 * 1. `-wdwClient:didEnterMonitoredArea:`
 * 2. `-wdwClient:didReceiveNewData:` (multiple times)
 * 3. `-wdwClient:didLeftMonitoredArea:`
 */
@protocol WDWClientDelegate <NSObject>

@optional
/**
 * At this point your App might want to establish a connection to the server. Called prior to
 * the `-wdwClient:didReceiveNewData:` callback
 */
- (void)wdwClient:(WDWClient *)client didEnterMonitoredArea:(NSString *)areaId;

/**
 * At this point your App might want to disconnect from the server.
 * The `-wdwClient:didReceiveNewData:` will no longer be called.
 */
- (void)wdwClient:(WDWClient *)client didLeftMonitoredArea:(NSString *)areaId;

@required
/// Used for handling errors. See SDK Error handling topic.
- (void)wdwClient:(WDWClient *)client didReceiveError:(NSError *)error;

/// The json data string is ready to be sent to the server. Do not modify the data.
/// This method is called repeatedly after the -wdwClient:didEnterMonitoredArea: callback has been triggered.
- (void)wdwClient:(WDWClient *)client didReceiveNewData:(NSString *)json;

/// Called when an incoming push message is recognized as a relevant driving warning for the current user
- (void)wdwClient:(WDWClient *)client didReceiveDrivingWarning:(WDWWarning *)warning;

@end

/**
 * Primary class to interact with the WDW-SDK. Use `WDWBuilder` to instanciate.
 */
@interface WDWClient : NSObject


/**
 * delegate should implement the `WDWClientDelegate` protocol.
 * All delegate methods are executed on the provided `delegateQueue`
 *
 */
@property (weak, nonatomic) id<WDWClientDelegate> delegate;

/**
 * Dispatch queue for the `WDWClientDelegate` callbacks execution
 *
 * By default, all delegate messages are delivered to the main dispatch queue, which helps you to manage UI update events. You can provide your own serial queue:
 *
 * <pre>self.wdwClient = dispatch_queue_create("myQueue", DISPATCH_QUEUE_SERIAL);</pre>
 */
@property (assign, nonatomic) dispatch_queue_t delegateQueue;



/**
 * Starts monitoring location and motion data. If the current location is in a relevant area, the WDW-SDK provides a json string. This string can be evaluated by the server. Also, the monitoring of the location is required for the evaluation of push messages.
 */
- (void)startMonitoring;

/**
 * Stops monitoring location and motion data. The monitoring can be restarted with `startMonitoring`.
 */
- (void)stopMonitoring;

/**
 * Evaluates the push notification payload. Triggers the `WDWClientDelegate` callback if a payload is recognized as a type of `WDWWarning`
 *
 * If you're using your own push instead of the WDW’s push service, the push message needs to be an NSDictionary using a message string as Key and a warning in form of an escaped JSON as value.
 *
 * <pre>
 *   NSDictionary *wdwMessage = @{@"message" = @"Put your escaped JSON String from your backend here"};
 *   [self.wdwClient evaluatePushMesage:wdwMessage];
 * </pre>
 *
 * If your delegate implements `-wdwClient:didReceiveDrivingWarning:` it will be triggered on the delegateQueue as result.
 *
 * @param payload you've received in AppDelegate's application:didReceiveRemoteNotification.
 *
 * @return `WDWWarning` object if notification sourced from WDW service, nil otherwise
 *
 * @see `-wdwClient:didReceiveDrivingWarning`
 */
- (WDWWarning *)evaluatePushMesage:(NSDictionary *)payload;

/**
 * Returns the WDW-SDK framework version
 */
+ (NSString *)version;

@end
