/**
 * Copyright (c) 2019 Bosch SoftTec GmbH All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Bosch
 * SoftTec GmbH ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Bosch SoftTec (BSOT).
 *
 * BSOT MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. BSOT SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

#import "AppDelegate.h"

@interface AppDelegate () <WDWClientDelegate>

@property (strong, nonatomic) NSData *token;

@end

@implementation AppDelegate

#pragma mark App Delegates

- (BOOL)application:(UIApplication *)application didFinishLaunchingWithOptions:(NSDictionary *)launchOptions {
    [application registerForRemoteNotifications]; //This wont work in simulator
    return YES;
}

- (void)application:(UIApplication *)application didRegisterForRemoteNotificationsWithDeviceToken:(NSData *)deviceToken {
    // Save received token, the device token is needed for registering for the WDW push service
    self.token = deviceToken;
    // Now ready to run the SDK
    [self buildSDK];
}

- (void)application:(UIApplication *)application didFailToRegisterForRemoteNotificationsWithError:(nonnull NSError *)error {
    NSLog(@"Failed to register for the push notification service: %@", error);
}

- (void)application:(UIApplication *)application didReceiveRemoteNotification:(NSDictionary *)userInfo fetchCompletionHandler:(void (^)(UIBackgroundFetchResult))completionHandler {
    
    // Forward all WDW push notifications for evaluation
    if ([self.wdwClient evaluatePushMesage:userInfo] == nil) {
        // handle application-related push message
        NSLog(@"Incoming push message is not related to the WDW");
    }
    completionHandler(UIBackgroundFetchResultNewData);
}

#pragma mark - Building WDW-SDK

- (WDWPushConfiguration *)pushConfiguration {
    return [[WDWPushConfiguration alloc]initWithPushToken:self.token
                                            server:@"Put Push Server URL here"
                                            variantID:@"Put Server Variant here"
                                            secret:@"Put Server Secret here"];
}

- (void)buildSDK {
    self.wdwClient = [[[[[WDWBuilder new]
                      withFleetID:@"Put Fleet ID here"]
                      withPushConfiguration:[self pushConfiguration]]
                      withLogLevel:WDWLogLevelInfo]
                      build];
    self.wdwClient.delegate = self;
}

#pragma mark - WDWClientDelegate

- (void)wdwClient:(WDWClient *)client didEnterMonitoredArea:(NSString *)areaId {
    // optional callback: get ready to receive data from the SDK
    // e.g. establish server connection
}

- (void)wdwClient:(WDWClient *)client didLeftMonitoredArea:(NSString *)areaId {
    // optional callback: get ready to stop receiving data from the SDK
    // e.g. close server connection
}

- (void)wdwClient:(WDWClient *)client didReceiveDrivingWarning:(WDWWarning *)warning {
    // display warning to user
    NSArray *messages = @[@"You're driving the wrong way!", @"There's a Wrong way driver nearby!", @"Everything looks OK now"];
    
    //Notify the user
    UIAlertController *alert = [UIAlertController alertControllerWithTitle:@"WDW" message:messages[warning.eventType] preferredStyle:UIAlertControllerStyleAlert];
       [alert addAction:[UIAlertAction actionWithTitle:@"OK" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
           [alert dismissViewControllerAnimated:YES completion:NULL];
       }]];
       [self.window.rootViewController presentViewController:alert animated:YES completion:NULL];
}

- (void)wdwClient:(WDWClient *)client didReceiveError:(NSError *)error {
    // handle errors appropriately
    switch (error.code) {
        case WDW_GPS_Access_Error:
            NSLog(@"ERROR - User does not allow location access");
            break;
        case WDW_Permissions_Error:
            NSLog(@"ERROR - User does not allow App to run in background");
            break;
        case WDW_UnsupportedHardware_Error:
            NSLog(@"ERROR - This device is not supported by the WDW-SDK");
            break;
        default:
            break;
    }
}

- (void)wdwClient:(WDWClient *)client didReceiveNewData:(NSString *)json {
    // send data to server
    NSLog(@"New data package from the WDW-SDK is available. Deliver it to your server");
}
@end
