/**
 * Copyright (c) 2017 Bosch SoftTec GmbH All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Bosch
 * SoftTec GmbH ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Bosch SoftTec (BSOT).
 *
 * BSOT MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. BSOT SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

#import <Foundation/Foundation.h>
#import "WDWPubDefinitions.h"

@class WDWClient, WDWWarning;


/**
 Delegate protocol to interact with the WDW-SDK.
 Delegate methods are only called when WDW-SDK monitoring is started.
 
 After the -[WDWClient startMonitoring] is called, the following delegate methods are called at arbitrary time:
 
  * -wdwClient:didChangeAvailability:
  * -wdwClient:didReceiveError:
  * -wdwClient:didReceiveDrivingWarning:
 
 Other delegate methods are executed in the following order:
 
 1. -wdwClient:didEnterMonitoredArea:
 2. -wdwClient:didReceiveNewData: (multiple times)
 3. -wdwClient:didLeftMonitoredArea:

 
 */
@protocol WDWClientDelegate <NSObject>

@optional
/** @brief At this point your App might want to establish a connection to the server. Called prior to
 * the -wdwClient:didReceiveNewData: callback
 */
- (void)wdwClient:(WDWClient *)client didEnterMonitoredArea:(NSString *)areaId;

/** @brief At this point your App might want to disconnect from the server.
 * The -wdwClient:didReceiveNewData: will no longer be called.
 */
- (void)wdwClient:(WDWClient *)client didLeftMonitoredArea:(NSString *)areaId;

/// @brief Called when an incoming push message is recognized as a relevant driving warning for the current user
- (void)wdwClient:(WDWClient *)client didReceiveDrivingWarning:(WDWWarning *)warning;

/// @brief Used for indicating the WDW-SDK availability to the user.
- (void)wdwClient:(WDWClient *)client didChangeAvailability:(WDWClientAvailability)availability;

@required
/// @brief Used for handling errors. See SDK Error handling topic.
- (void)wdwClient:(WDWClient *)client didReceiveError:(NSError *)error;

/// @brief The json data string is ready to be sent to the server. Do not modify the data.
/// This method called repeatedly after the -wdwClient:didEnterMonitoredArea: callback has been triggered.
- (void)wdwClient:(WDWClient *)client didReceiveNewData:(NSString *)json;
@end

/**
 * Primary class to interact with the WDW-SDK. Use WDWBuilder to instanciate.
 */
@interface WDWClient : NSObject


/** @brief delegate should implement the WDWClientDelegate protocol.
 * All delegate methods are executed on the provided delegateQueue
 
 */
@property (weak, nonatomic) id<WDWClientDelegate> delegate;

/// @brief Presents current state of the WDWClient
/// @see WDWClientAvailability
/// @deprecated The availablity feature is deprecated from version 3.2.0
@property (assign, nonatomic, readonly) WDWClientAvailability availability DEPRECATED_MSG_ATTRIBUTE("The availablity feature is deprecated from version 3.2.0");
/**
 @brief Dispatch queue for the WDWClientDelegate callbacks execution
 
 @discussion By default, all delegate messages are delivered to the main dispatch queue, which helps you to manage UI update events. You can provide your own serial queue:
 
    self.wdwClient = dispatch_queue_create("myQueue", DISPATCH_QUEUE_SERIAL);
 */
@property (assign, nonatomic) dispatch_queue_t delegateQueue;



/**
 *  @brief Starts monitoring location and motion data. If the current location is in a relevant area, the WDW-SDK provides a json string. This string can be evaluated by the server. Also, the monitoring of the location is required for the evaluation of push message.
 */
- (void)startMonitoring;

/**
 * @brief Stops monitoring location and motion data. The monitoring can be restarted with startMonitoring.
 */
- (void)stopMonitoring;


/** 
 * @brief Evaluates the push notification payload. Triggers the WDWClientDelegate callback if a payload
 * is recognized as a type of WDWWarning
 * @note If your delegate implements -wdwClient:didReceiveDrivingWarning: it will be triggered on the delegateQueue as result
 * @param payload you've received in AppDelegate's -application:didReceiveRemoteNotification:
 * @return WDWWarning object if notification sourced from WDW service, nil otherwise
 * @see -wdwClient:didReceiveDrivingWarning:
 */
- (WDWWarning *)evaluatePushMesage:(NSDictionary *)payload;

/**
 * @brief Returns the WDWSDK framework version
 */
+ (NSString *)version;

@end
