/*
 * Copyright (c) 2017 Bosch SoftTec GmbH All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Bosch
 * SoftTec GmbH ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Bosch SoftTec (BSOT).
 *
 * BSOT MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. BSOT SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

#import <Foundation/Foundation.h>

@class WDWWarning;

/** Specifies the WDW-SDK messages evaluation protocol. Allows you to implement your own message evaluation routine for incoming messages.

 */
@protocol WDWMessageEvaluation <NSObject>

/** 
 @brief Implement this method for custom WDW-SDK message evaluation
 @param object Push notification object which is received in AppDelegate's -application:didReceiveRemoteNotification:
 
 #Parsing the notification object
 You need to extract the WDW event object at the key path "aps.alert.body":
 
    NSDictionary *wdwMessage = [object valueForKeyPath:@"aps.alert.body"];
 
 Then analyze and evaluate it accordingly to the specified data structure.
 
##Content of wdwMessage object
 | key     | json type | content                                       |
 |:--------|:---------:|:----------------------------------------------|
 | meta    | object    | Metadata of the Wrong Way Driver push service |
 | data    | object    | Data of the Wrong Way Driver push service     |
 | location| object    | Location of the Wrong Way Driver event        |
 
 ##Content of Metadata
 | key             | json type | content                                                         |
 |:----------------|:---------:|:----------------------------------------------------------------|
 | radius          | Number    | The relevant radius of the Wrong Way Driver event, meters       |
 | lifetime        | Number    | Lifespan of the Wrong Way Driver event till it expires, seconds |
 | headingDirection| Number    | Heading of the Wrong Way Driver event, degree                   |
 | headingRelevance| Number    | Heading relevance of the Wrong Way Driver event                 |
 
 ##Content of Data
 | key             | json type     | content                                                  |
 |:----------------|:-------------:|:---------------------------------------------------------|
 | driveID         | String        | Unique identifier of the drive which has caused the event|
 | event           | String        | Action event                                             |
 | uniqueId        | String        | The unique identifier of the Wrong Way Driver event      |
 | timeGenerated   | Number        | Time, when the Wrong Way Driver event took place, POSIX timestamp|
 | tags            | Array[String] | Tags for checking the relevance for the device           |
 
 ##Content of Location
 | key               | json type | content                        |
 |:------------------|:---------:|:-------------------------------|
 | latitude          | Number    | GPS latitude in degrees        |
 | longitude         | Number    | GPS longitude in degrees       |
 | altitude          | Number    | GPS altitude in m              |
 | horizontalAccuracy| Number    | GPS horizontalAccuracy, meters |
 | verticalAccuracy  | Number    | GPS verticalAccuracy, meters   |
 | heading           | Number    | GPS heading, grad              |
 | speed             | Number    | GPS speed, m/s                 |
 
@return WDWWarning object with properly recognized eventType or nil.
 
 You should return nil if the Wrong Way Driver event dictionary passed by the object argument lacks at least one of the required fields (meta, data, location).
 Otherwise, you should properly calculate the eventType, a distance to it and return them as a WDWWarning instance:
 
    if (!wdwEvent[@"meta"] || !wdwEvent[@"data"] || !wdwEvent[@"location"]) return nil;
    // your evaluation code
    // ...
    return [[WDWWarning alloc] initWithEventType:calculatedType andDistance:calculatedDistance];
 */
- (WDWWarning *)evaluateMessage:(NSDictionary *)object;


- (void)startEvaluating;

- (void)stopEvaluating;

@end


