/**
 * Copyright (c) 2017 Bosch SoftTec GmbH All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Bosch
 * SoftTec GmbH ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Bosch SoftTec (BSOT).
 *
 * BSOT MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. BSOT SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

#import "AppDelegate.h"
#import <CoreLocation/CoreLocation.h>

@interface AppDelegate () <WDWClientDelegate>

@property (strong, nonatomic) NSData *token;

@end

@implementation AppDelegate

#pragma mark - WDWClientDelegate

- (void)wdwClient:(WDWClient *)client didEnterMonitoredArea:(NSString *)areaId {
    // optional callback: get ready to receive data from the SDK
    // e.g. establish server connection
}

- (void)wdwClient:(WDWClient *)client didLeftMonitoredArea:(NSString *)areaId {
    // optional callback: get ready to stop receiving data from the SDK
    // e.g. close server connection
}

- (void)wdwClient:(WDWClient *)client didReceiveDrivingWarning:(WDWWarning *)warning {
    // display warning to user
    
    NSArray *messages = @[@"I'am driving the wrong way!", @"Wrong way driver on my way!", @"Everything looks OK now"];
    // Notify the UI
    [[NSNotificationCenter defaultCenter] postNotificationName:@"WDWWarningReceived" object:messages[warning.eventType]];
}

- (void)wdwClient:(WDWClient *)client didChangeAvailability:(WDWClientAvailability)availability {
    NSLog(@"--------- Availability");
    // notify user of WDW-SDK availability
    
    NSArray *availabityStrings = @[@"Unavailable", @"Temporary Unavailable", @"Available"];
    // Notify the UI
    [[NSNotificationCenter defaultCenter] postNotificationName:@"WDWAvailabilityChanged" object:availabityStrings[availability]];
}

- (void)wdwClient:(WDWClient *)client didReceiveError:(NSError *)error {
    // handle errors
    
    switch (error.code) {
        case WDW_GPS_Access_Error:
            NSLog(@"User does not allow location access");
            [self showError:error];
            break;
        case WDW_Permissions_Error:
            NSLog(@"User does not allow App to run in background");
            [self showError:error];
            break;
        case WDW_UnsupportedHardware_Error:
            NSLog(@"This device is not supported by the WDW-SDK");
            break;
        default:
            break;
    }
    
}

- (void)wdwClient:(WDWClient *)client didReceiveNewData:(NSString *)json {
    // send data to server
    
    NSLog(@"New data package available. Deliver it to your server");
}


#pragma mark Forwarding push message
- (void)application:(UIApplication *)application didReceiveRemoteNotification:(NSDictionary *)userInfo fetchCompletionHandler:(void (^)(UIBackgroundFetchResult))completionHandler {
    
    // Forward all push notifications for evaluation
    if ([self.wdwClient evaluatePushMesage:userInfo] == nil) {
        NSLog(@"Incoming push message is not recognized as WDW event");
        // Apply processing for your own push message type
    }
    
    completionHandler(UIBackgroundFetchResultNewData);
}

#pragma mark - Building SDK
- (WDWPushConfiguration *)pushConfiguration {
    return [[WDWPushConfiguration alloc] initWithPushToken:self.token
                                                    server:@"Enter the Push Server URL"
                                                 variantID:@"Enter Variant"
                                                    secret:@"Enter Secret"
            ];
}

- (void)buildSDK {
    
    self.wdwClient = [[[[[WDWBuilder new]
                      withFleetID:@"Enter your Fleet ID"]
                      withPushConfiguration:[self pushConfiguration]]
                      withLogLevel:WDWLogLevelInfo]
                      build];
    
    self.wdwClient.delegate = self;
}

#pragma mark device token
- (BOOL)application:(UIApplication *)application didFinishLaunchingWithOptions:(NSDictionary *)launchOptions {

    // Push notifiactions are not suported by iOS simulator platform
#if !TARGET_IPHONE_SIMULATOR
    [application registerForRemoteNotifications];
#else
    // mock device token
    self.token = [NSData data];
    // build the SDK
    [self buildSDK];
#endif
    return YES;
}

- (void)application:(UIApplication *)application didRegisterForRemoteNotificationsWithDeviceToken:(NSData *)deviceToken {
    // Save received token
    self.token = deviceToken;
    // Now ready to run the SDK
    [self buildSDK];
    
    
}

- (void)application:(UIApplication *)application didFailToRegisterForRemoteNotificationsWithError:(nonnull NSError *)error {
    NSLog(@"Failed to register at push notification service: %@", error);
}


#pragma mark -


- (void)showError:(NSError *)error {
    // handle errors
    
    UIAlertController *alert = [UIAlertController alertControllerWithTitle:nil message:[error localizedDescription] preferredStyle:UIAlertControllerStyleAlert];
    
    [alert addAction:[UIAlertAction actionWithTitle:@"OK" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        [alert dismissViewControllerAnimated:YES completion:NULL];
    }]];
    
    [self.window.rootViewController presentViewController:alert animated:YES completion:NULL];
}


@end
